/* /designer/designer.js
   Minimal Konva-based panel designer for the PHP API endpoints.

   Assumptions:
   - This file is served from /designer/
   - API is at /api/...
   - Uploads served from /uploads/...
*/

const API = {
  createProject: '/api/projects/create.php',
  getProject: '/api/projects/get.php',
  saveItems: '/api/projects/save_items.php',

  uploadBg: '/api/background/upload.php',
  saveCorrected: '/api/background/save_corrected.php',
  saveCalibration: '/api/background/save_calibration.php',

  listAssets: '/api/assets/list.php',

  exportSave: '/api/export/save.php',
  bom: '/api/export/bom.php'
};

const el = (id) => document.getElementById(id);

const ui = {
  projectName: el('projectName'),
  projectToken: el('projectToken'),

  btnCreateProject: el('btnCreateProject'),
  btnOpenProject: el('btnOpenProject'),
  btnReloadProject: el('btnReloadProject'),

  bgMode: el('bgMode'),
  bgFile: el('bgFile'),
  btnUploadBg: el('btnUploadBg'),
  btnPickCorners: el('btnPickCorners'),
  btnUploadCorrected: el('btnUploadCorrected'),

  knownMM: el('knownMM'),
  pxPerMM: el('pxPerMM'),
  btnCalibrate: el('btnCalibrate'),
  btnSaveScale: el('btnSaveScale'),

  btnRotateL: el('btnRotateL'),
  btnRotateR: el('btnRotateR'),
  btnDeleteSel: el('btnDeleteSel'),

  btnSaveItems: el('btnSaveItems'),
  btnExportBefore: el('btnExportBefore'),
  btnExportAfter: el('btnExportAfter'),
  btnBOM: el('btnBOM'),

  assetCategory: el('assetCategory'),
  assetSearch: el('assetSearch'),
  btnLoadAssets: el('btnLoadAssets'),
  assetList: el('assetList'),

  output: el('output'),
  statusBadge: el('statusBadge'),
  toast: el('toast'),
};

let state = {
  token: '',
  projectId: 0,
  projectName: 'Untitled',

  backgroundId: 0,
  backgroundMode: 'straight',
  bgOriginalUrl: '',
  bgCorrectedUrl: '',

  // scaling
  pxPerMM: 0,

  // assets
  assets: [], // from API
  assetImgCache: new Map(), // asset_id -> HTMLImageElement

  // konva
  stage: null,
  bgLayer: null,
  itemLayer: null,
  uiLayer: null,
  transformer: null,

  bgKonvaImage: null, // Konva.Image
  bgBitmap: null,     // HTMLImageElement used for sampling/warp

  // interactions
  mode: 'idle', // idle | pickCorners | calibrate
  cornerPoints: [],   // [{x,y}]
  calPoints: [],      // [{x,y}]
  isPanning: false,
  panStart: null,

  // placed items tracking
  itemNodes: new Map(), // nodeId -> { dbId, asset_id, meta }
  dbIdByNodeId: new Map(),
};

function log(msg) {
  ui.output.textContent = (ui.output.textContent ? ui.output.textContent + "\n" : "") + msg;
}

function toast(msg, ms=2200) {
  ui.toast.textContent = msg;
  ui.toast.style.display = 'block';
  clearTimeout(toast._t);
  toast._t = setTimeout(() => ui.toast.style.display = 'none', ms);
}

function setStatus(text) {
  ui.statusBadge.textContent = text;
}

function enableControls(enabled) {
  ui.btnSaveItems.disabled = !enabled;
  ui.btnExportBefore.disabled = !enabled;
  ui.btnExportAfter.disabled = !enabled;
  ui.btnBOM.disabled = !enabled;
}

function selectionControls(enabled) {
  ui.btnRotateL.disabled = !enabled;
  ui.btnRotateR.disabled = !enabled;
  ui.btnDeleteSel.disabled = !enabled;
}

/* ---------------------------
   Konva setup
--------------------------- */
function initStage() {
  const wrap = el('stageWrap');
  wrap.innerHTML = '';

  const w = wrap.clientWidth;
  const h = wrap.clientHeight;

  const stage = new Konva.Stage({
    container: 'stageWrap',
    width: w,
    height: h,
    draggable: false
  });

  const bgLayer = new Konva.Layer();
  const itemLayer = new Konva.Layer();
  const uiLayer = new Konva.Layer();

  stage.add(bgLayer);
  stage.add(itemLayer);
  stage.add(uiLayer);

  const transformer = new Konva.Transformer({
    rotateEnabled: false,       // we handle rotate via buttons (v1)
    enabledAnchors: [],         // lock scaling; size controlled by pxPerMM
    borderStroke: 'rgba(147,197,253,.95)',
    borderDash: [6, 6],
  });
  uiLayer.add(transformer);

  stage.on('mousedown touchstart', (e) => onStagePointerDown(e));
  stage.on('mousemove touchmove', (e) => onStagePointerMove(e));
  stage.on('mouseup touchend',   (e) => onStagePointerUp(e));

  // resize
  window.addEventListener('resize', () => {
    const nw = wrap.clientWidth;
    const nh = wrap.clientHeight;
    stage.size({ width: nw, height: nh });
    stage.draw();
  });

  state.stage = stage;
  state.bgLayer = bgLayer;
  state.itemLayer = itemLayer;
  state.uiLayer = uiLayer;
  state.transformer = transformer;

  enableControls(false);
  selectionControls(false);
}
initStage();

/* ---------------------------
   Helpers for pointer coords
--------------------------- */
function stagePointerPos() {
  const pos = state.stage.getPointerPosition();
  if (!pos) return null;
  // account for stage drag (we use manual pan via position)
  const x = (pos.x - state.stage.x()) / state.stage.scaleX();
  const y = (pos.y - state.stage.y()) / state.stage.scaleY();
  return { x, y };
}

/* ---------------------------
   Pan + selection + modes
--------------------------- */
function onStagePointerDown(e) {
  const isShift = e.evt && e.evt.shiftKey;
  const isRight = e.evt && e.evt.button === 2;

  if (isShift || isRight) {
    state.isPanning = true;
    state.panStart = { x: e.evt.clientX, y: e.evt.clientY, sx: state.stage.x(), sy: state.stage.y() };
    return;
  }

  const pos = stagePointerPos();
  if (!pos) return;

  // Mode-specific click handling
  if (state.mode === 'pickCorners') {
    addCornerPoint(pos);
    return;
  }

  if (state.mode === 'calibrate') {
    addCalPoint(pos);
    return;
  }

  // Normal selection
  const clickedOnEmpty = e.target === state.stage || e.target === state.bgKonvaImage;
  if (clickedOnEmpty) {
    state.transformer.nodes([]);
    selectionControls(false);
    state.uiLayer.draw();
  }
}

function onStagePointerMove(e) {
  if (!state.isPanning || !state.panStart) return;
  const dx = e.evt.clientX - state.panStart.x;
  const dy = e.evt.clientY - state.panStart.y;
  state.stage.position({ x: state.panStart.sx + dx, y: state.panStart.sy + dy });
  state.stage.batchDraw();
}

function onStagePointerUp(e) {
  state.isPanning = false;
  state.panStart = null;
}

// Prevent context menu for right-drag pan
document.addEventListener('contextmenu', (e) => e.preventDefault());

/* ---------------------------
   Background loading into Konva
--------------------------- */
function setBackgroundImage(url) {
  return new Promise((resolve, reject) => {
    const img = new Image();
    img.crossOrigin = 'anonymous';
    img.onload = () => resolve(img);
    img.onerror = reject;
    img.src = url + (url.includes('?') ? '&' : '?') + 't=' + Date.now();
  }).then((img) => {
    state.bgBitmap = img;

    // Clear old
    state.bgLayer.destroyChildren();
    state.bgKonvaImage = new Konva.Image({
      image: img,
      x: 0,
      y: 0
    });
    state.bgLayer.add(state.bgKonvaImage);

    // Reset view: fit to stage
    fitStageToBackground(img.width, img.height);

    state.bgLayer.draw();
    state.itemLayer.draw();
    state.uiLayer.draw();
  });
}

function fitStageToBackground(imgW, imgH) {
  const wrap = el('stageWrap');
  const stageW = wrap.clientWidth;
  const stageH = wrap.clientHeight;

  const scale = Math.min(stageW / imgW, stageH / imgH);
  state.stage.scale({ x: scale, y: scale });
  state.stage.position({
    x: (stageW - imgW * scale) / 2,
    y: (stageH - imgH * scale) / 2
  });
  state.stage.batchDraw();
}

/* ---------------------------
   Corner picking + warp
--------------------------- */
function addCornerPoint(pos) {
  if (!state.bgBitmap) { toast('Upload/load a background first.'); return; }

  if (state.cornerPoints.length >= 4) return;

  state.cornerPoints.push({ x: pos.x, y: pos.y });

  // draw marker
  const r = new Konva.Circle({
    x: pos.x,
    y: pos.y,
    radius: 8,
    fill: 'rgba(245,158,11,.85)',
    stroke: 'rgba(255,255,255,.7)',
    strokeWidth: 2
  });
  const t = new Konva.Text({
    x: pos.x + 10,
    y: pos.y - 10,
    text: String(state.cornerPoints.length),
    fontSize: 16,
    fill: 'rgba(245,158,11,.95)',
    fontStyle: 'bold'
  });

  state.uiLayer.add(r);
  state.uiLayer.add(t);
  state.uiLayer.draw();

  if (state.cornerPoints.length === 4) {
    toast('4 corners selected. Now upload corrected.');
    ui.btnUploadCorrected.disabled = false;
    state.mode = 'idle';
    ui.btnPickCorners.disabled = false;
  }
}

function clearCornerUI() {
  // remove circles/texts used for corners (cheap: rebuild ui layer except transformer)
  const keep = [state.transformer];
  state.uiLayer.destroyChildren();
  keep.forEach(k => state.uiLayer.add(k));
  state.cornerPoints = [];
  state.uiLayer.draw();
}

function estimateWarpSize(pts) {
  // pts in order: TL, TR, BR, BL
  const d = (a,b) => Math.hypot(a.x-b.x, a.y-b.y);
  const top = d(pts[0], pts[1]);
  const bottom = d(pts[3], pts[2]);
  const left = d(pts[0], pts[3]);
  const right = d(pts[1], pts[2]);
  let w = Math.max(top, bottom);
  let h = Math.max(left, right);

  // clamp for performance
  const maxSide = 1800;
  const scale = Math.min(1, maxSide / Math.max(w, h));
  w = Math.max(300, Math.round(w * scale));
  h = Math.max(300, Math.round(h * scale));
  return { w, h };
}

/**
 * Compute homography H mapping src->dst (both 4 points).
 * Returns 3x3 matrix as flat array length 9 (row-major).
 *
 * Method: solve 8x8 linear system for h11..h32 with h33=1.
 */
function homographyFrom4pt(src, dst) {
  // Build A x = b
  const A = [];
  const b = [];
  for (let i=0; i<4; i++) {
    const x = src[i].x, y = src[i].y;
    const u = dst[i].x, v = dst[i].y;

    // u = (h11 x + h12 y + h13) / (h31 x + h32 y + 1)
    // v = (h21 x + h22 y + h23) / (h31 x + h32 y + 1)
    A.push([x, y, 1, 0, 0, 0, -u*x, -u*y]); b.push(u);
    A.push([0, 0, 0, x, y, 1, -v*x, -v*y]); b.push(v);
  }

  const x = solveLinearSystem8(A, b); // length 8
  const H = [
    x[0], x[1], x[2],
    x[3], x[4], x[5],
    x[6], x[7], 1
  ];
  return H;
}

// Gaussian elimination for 8x8
function solveLinearSystem8(A, b) {
  // deep copy
  const M = A.map(row => row.slice());
  const B = b.slice();
  const n = 8;

  for (let col=0; col<n; col++) {
    // pivot
    let pivotRow = col;
    let maxVal = Math.abs(M[col][col]);
    for (let r=col+1; r<n; r++) {
      const v = Math.abs(M[r][col]);
      if (v > maxVal) { maxVal = v; pivotRow = r; }
    }
    if (maxVal < 1e-12) throw new Error('Degenerate points (cannot solve homography).');

    // swap
    if (pivotRow !== col) {
      [M[col], M[pivotRow]] = [M[pivotRow], M[col]];
      [B[col], B[pivotRow]] = [B[pivotRow], B[col]];
    }

    // normalize pivot row
    const piv = M[col][col];
    for (let c=col; c<n; c++) M[col][c] /= piv;
    B[col] /= piv;

    // eliminate
    for (let r=0; r<n; r++) {
      if (r === col) continue;
      const factor = M[r][col];
      if (Math.abs(factor) < 1e-12) continue;
      for (let c=col; c<n; c++) M[r][c] -= factor * M[col][c];
      B[r] -= factor * B[col];
    }
  }
  return B;
}

function invert3x3(H) {
  const a=H[0], b=H[1], c=H[2];
  const d=H[3], e=H[4], f=H[5];
  const g=H[6], h=H[7], i=H[8];

  const A = e*i - f*h;
  const B = f*g - d*i;
  const C = d*h - e*g;
  const D = c*h - b*i;
  const E = a*i - c*g;
  const F = b*g - a*h;
  const G = b*f - c*e;
  const Hh = c*d - a*f;
  const I = a*e - b*d;

  const det = a*A + b*B + c*C;
  if (Math.abs(det) < 1e-12) throw new Error('Non-invertible homography.');

  const invDet = 1 / det;
  return [
    A*invDet, D*invDet, G*invDet,
    B*invDet, E*invDet, Hh*invDet,
    C*invDet, F*invDet, I*invDet
  ];
}

function applyHomography(H, x, y) {
  const nx = H[0]*x + H[1]*y + H[2];
  const ny = H[3]*x + H[4]*y + H[5];
  const nz = H[6]*x + H[7]*y + H[8];
  return { x: nx/nz, y: ny/nz };
}

/**
 * Warp current bgBitmap using selected cornerPoints to a rectified image.
 * Returns a Blob (PNG) and also sets corrected background on stage.
 */
async function warpBackgroundToBlob() {
  if (!state.bgBitmap) throw new Error('No background loaded.');
  if (state.cornerPoints.length !== 4) throw new Error('Need 4 corner points.');

  // Expect order TL, TR, BR, BL
  const src = state.cornerPoints.map(p => ({x:p.x, y:p.y}));
  const size = estimateWarpSize(src);

  const dst = [
    {x:0, y:0},
    {x:size.w-1, y:0},
    {x:size.w-1, y:size.h-1},
    {x:0, y:size.h-1},
  ];

  // H maps src -> dst, so inverse maps dst -> src for sampling
  const H = homographyFrom4pt(src, dst);
  const Hinv = invert3x3(H);

  // Offscreen canvases
  const srcCanvas = document.createElement('canvas');
  srcCanvas.width = state.bgBitmap.width;
  srcCanvas.height = state.bgBitmap.height;
  const sctx = srcCanvas.getContext('2d', { willReadFrequently: true });
  sctx.drawImage(state.bgBitmap, 0, 0);

  const srcData = sctx.getImageData(0, 0, srcCanvas.width, srcCanvas.height);
  const sw = srcCanvas.width, sh = srcCanvas.height;

  const outCanvas = document.createElement('canvas');
  outCanvas.width = size.w;
  outCanvas.height = size.h;
  const octx = outCanvas.getContext('2d', { willReadFrequently: true });
  const outImg = octx.createImageData(size.w, size.h);
  const out = outImg.data;
  const inD = srcData.data;

  // nearest neighbor sample (fast, v1 acceptable)
  for (let y=0; y<size.h; y++) {
    for (let x=0; x<size.w; x++) {
      const sp = applyHomography(Hinv, x, y);
      const sx = Math.round(sp.x);
      const sy = Math.round(sp.y);

      const oi = (y*size.w + x) * 4;
      if (sx >= 0 && sx < sw && sy >= 0 && sy < sh) {
        const si = (sy*sw + sx) * 4;
        out[oi]   = inD[si];
        out[oi+1] = inD[si+1];
        out[oi+2] = inD[si+2];
        out[oi+3] = inD[si+3];
      } else {
        // transparent
        out[oi] = 0; out[oi+1] = 0; out[oi+2] = 0; out[oi+3] = 0;
      }
    }
  }

  octx.putImageData(outImg, 0, 0);

  // Convert to blob
  const blob = await new Promise((resolve) => outCanvas.toBlob(resolve, 'image/png', 0.92));
  if (!blob) throw new Error('Failed to create corrected image blob.');

  // Preview corrected immediately (local)
  const localUrl = URL.createObjectURL(blob);
  await setBackgroundImage(localUrl);

  // Store corner json to send to backend
  state._lastCornerPointsJson = JSON.stringify(state.cornerPoints);

  return blob;
}

/* ---------------------------
   Calibration
--------------------------- */
function addCalPoint(pos) {
  if (!state.bgBitmap) { toast('Load background first.'); return; }

  if (state.calPoints.length >= 2) return;

  state.calPoints.push({ x: pos.x, y: pos.y });

  const r = new Konva.Circle({
    x: pos.x,
    y: pos.y,
    radius: 7,
    fill: 'rgba(34,197,94,.9)',
    stroke: 'rgba(255,255,255,.7)',
    strokeWidth: 2
  });
  state.uiLayer.add(r);

  if (state.calPoints.length === 2) {
    const a = state.calPoints[0], b = state.calPoints[1];
    const line = new Konva.Line({
      points: [a.x, a.y, b.x, b.y],
      stroke: 'rgba(34,197,94,.95)',
      strokeWidth: 3
    });
    state.uiLayer.add(line);
    state.uiLayer.draw();

    state.mode = 'idle';
    ui.btnCalibrate.disabled = false;

    // compute px/mm if known provided
    const known = parseFloat(ui.knownMM.value || '0');
    if (known > 0) {
      const px = Math.hypot(b.x - a.x, b.y - a.y);
      const ppm = px / known;
      ui.pxPerMM.value = ppm.toFixed(6);
      state.pxPerMM = ppm;
      toast(`Scale computed: ${ppm.toFixed(6)} px/mm`);
      ui.btnSaveScale.disabled = false;
    } else {
      toast('Enter known mm to compute px/mm.');
    }
  } else {
    state.uiLayer.draw();
  }
}

function clearCalUI() {
  // rebuild ui layer except transformer
  const keep = [state.transformer];
  state.uiLayer.destroyChildren();
  keep.forEach(k => state.uiLayer.add(k));
  state.calPoints = [];
  state.uiLayer.draw();
}

/* ---------------------------
   Assets UI + drag/drop
--------------------------- */
async function loadAssets() {
  const params = new URLSearchParams();
  params.set('active', '1');
  if (ui.assetCategory.value.trim()) params.set('category', ui.assetCategory.value.trim());
  if (ui.assetSearch.value.trim()) params.set('q', ui.assetSearch.value.trim());

  const r = await fetch(`${API.listAssets}?${params.toString()}`);
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Failed to load assets.');

  state.assets = j.assets || [];
  renderAssetList();
  toast(`Loaded ${state.assets.length} assets`);
}

function renderAssetList() {
  ui.assetList.innerHTML = '';
  for (const a of state.assets) {
    const div = document.createElement('div');
    div.className = 'asset';
    div.draggable = true;
    div.dataset.assetId = String(a.id);

    const img = document.createElement('img');
    img.src = a.file_path;
    img.alt = a.name;

    const nm = document.createElement('div');
    nm.className = 'name';
    nm.textContent = a.name;

    const meta = document.createElement('div');
    meta.className = 'meta';
    meta.textContent = `${a.category} • ${a.width_mm}×${a.height_mm} mm`;

    div.appendChild(img);
    div.appendChild(nm);
    div.appendChild(meta);

    div.addEventListener('dragstart', (ev) => {
      ev.dataTransfer.setData('text/plain', JSON.stringify({
        asset_id: a.id
      }));
    });

    ui.assetList.appendChild(div);
  }
}

// Allow dropping into stage container
el('stageWrap').addEventListener('dragover', (e) => {
  e.preventDefault();
});
el('stageWrap').addEventListener('drop', async (e) => {
  e.preventDefault();
  if (!state.token) return toast('Create/open a project first.');
  if (!state.bgBitmap) return toast('Upload/load a background first.');
  if (!state.pxPerMM || state.pxPerMM <= 0) return toast('Calibrate scale first (px/mm).');

  const data = e.dataTransfer.getData('text/plain');
  if (!data) return;
  let parsed;
  try { parsed = JSON.parse(data); } catch { return; }
  const assetId = parseInt(parsed.asset_id, 10);
  if (!assetId) return;

  // compute stage coords
  const rect = el('stageWrap').getBoundingClientRect();
  const sx = (e.clientX - rect.left - state.stage.x()) / state.stage.scaleX();
  const sy = (e.clientY - rect.top - state.stage.y()) / state.stage.scaleY();

  await placeAsset(assetId, sx, sy);
});

async function getAssetById(id) {
  return state.assets.find(a => Number(a.id) === Number(id));
}

async function loadAssetImage(asset) {
  if (state.assetImgCache.has(asset.id)) return state.assetImgCache.get(asset.id);

  const img = new Image();
  img.crossOrigin = 'anonymous';

  await new Promise((resolve, reject) => {
    img.onload = resolve;
    img.onerror = reject;
    img.src = asset.file_path + (asset.file_path.includes('?') ? '&' : '?') + 't=' + Date.now();
  });

  state.assetImgCache.set(asset.id, img);
  return img;
}

async function placeAsset(assetId, x, y) {
  const asset = await getAssetById(assetId);
  if (!asset) return toast('Asset not found (reload assets).');

  const img = await loadAssetImage(asset);

  const wPx = Number(asset.width_mm) * state.pxPerMM;
  const hPx = Number(asset.height_mm) * state.pxPerMM;

  const node = new Konva.Image({
    image: img,
    x: x - wPx/2,
    y: y - hPx/2,
    width: wPx,
    height: hPx,
    draggable: true
  });

  // Selection
  node.on('mousedown touchstart', () => {
    state.transformer.nodes([node]);
    selectionControls(true);
    state.uiLayer.draw();
  });

  // Keep inside optional bounds? (v1: no)
  node.on('dragend', () => { /* no-op */ });

  state.itemLayer.add(node);
  state.itemLayer.draw();

  // Track mapping
  const nodeId = node._id; // internal Konva id
  state.itemNodes.set(nodeId, { dbId: 0, asset_id: assetId, meta: null });

  enableControls(true);
}

/* ---------------------------
   Save/Load project
--------------------------- */
function setToken(token) {
  state.token = token || '';
  ui.projectToken.value = state.token;
  if (state.token) {
    localStorage.setItem('panel_designer_token', state.token);
    setStatus(`Project: ${state.token}`);
  } else {
    setStatus('No project');
  }
}

function setProjectName(name) {
  state.projectName = name || 'Untitled';
  ui.projectName.value = state.projectName;
}

async function createProject() {
  const name = ui.projectName.value.trim() || 'Untitled';
  const r = await fetch(API.createProject, {
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify({ name })
  });
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Create project failed.');
  setToken(j.project_token);
  state.projectId = j.project_id;
  toast('Project created');
  enableControls(true);

  // Put token in URL
  const url = new URL(location.href);
  url.searchParams.set('token', state.token);
  history.replaceState({}, '', url.toString());
}

async function openProjectFromToken(token) {
  setToken(token);
  await reloadProject();
}

async function reloadProject() {
  if (!state.token) return toast('No token.');
  const r = await fetch(`${API.getProject}?token=${encodeURIComponent(state.token)}`);
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Load project failed.');

  state.projectId = j.project?.id || 0;
  setProjectName(j.project?.name || 'Untitled');

  // Clear layers/items
  state.itemLayer.destroyChildren();
  state.itemNodes.clear();
  state.transformer.nodes([]);
  selectionControls(false);

  // Background
  state.backgroundId = j.background?.id || 0;
  state.backgroundMode = j.background?.mode || 'straight';
  ui.bgMode.value = state.backgroundMode;

  state.bgOriginalUrl = j.background?.original_path || '';
  state.bgCorrectedUrl = j.background?.corrected_path || '';

  // choose best background url
  const bgUrl = state.bgCorrectedUrl || state.bgOriginalUrl;
  if (bgUrl) {
    await setBackgroundImage(bgUrl);
    enableBgDependentUI(true);
  } else {
    enableBgDependentUI(false);
  }

  // scale
  const ppm = j.background?.px_per_mm ? Number(j.background.px_per_mm) : 0;
  state.pxPerMM = ppm > 0 ? ppm : 0;
  ui.pxPerMM.value = state.pxPerMM ? state.pxPerMM.toFixed(6) : '';

  ui.btnSaveScale.disabled = !(state.backgroundId > 0);
  ui.btnCalibrate.disabled = !(state.backgroundId > 0);

  // Items
  const items = j.items || [];
  // Ensure assets are loaded for proper placement (optional: load lazily)
  if (!state.assets.length) {
    try { await loadAssets(); } catch { /* ignore */ }
  }

  for (const it of items) {
    await placeItemFromDB(it);
  }

  enableControls(true);
  toast('Project loaded');
}

function enableBgDependentUI(hasBg) {
  ui.btnPickCorners.disabled = !hasBg || (ui.bgMode.value !== 'perspective');
  ui.btnUploadCorrected.disabled = true;
  ui.btnCalibrate.disabled = !hasBg;
  ui.btnSaveScale.disabled = !hasBg;
}

async function placeItemFromDB(it) {
  const assetId = Number(it.asset_id);
  const x = Number(it.x_px);
  const y = Number(it.y_px);
  const rot = Number(it.rotation_deg || 0);

  const asset = await getAssetById(assetId);
  if (!asset) return; // if missing from assets list, skip silently

  const img = await loadAssetImage(asset);

  const wPx = Number(asset.width_mm) * state.pxPerMM;
  const hPx = Number(asset.height_mm) * state.pxPerMM;

  const node = new Konva.Image({
    image: img,
    x,
    y,
    width: wPx,
    height: hPx,
    draggable: true,
    rotation: rot
  });

  node.on('mousedown touchstart', () => {
    state.transformer.nodes([node]);
    selectionControls(true);
    state.uiLayer.draw();
  });

  state.itemLayer.add(node);
  state.itemLayer.draw();

  const nodeId = node._id;
  state.itemNodes.set(nodeId, { dbId: Number(it.id), asset_id: assetId, meta: it.meta_json ? safeJsonParse(it.meta_json) : null });
}

function safeJsonParse(s) {
  try { return JSON.parse(s); } catch { return null; }
}

async function saveItems() {
  if (!state.token) return toast('No project token.');
  const payload = {
    token: state.token,
    items: [],
    deleted_ids: [] // v1: deletions handled immediately by delete button and tracked in memory below
  };

  // gather items
  const nodes = state.itemLayer.getChildren(node => node.className === 'Image');
  let z = 0;
  for (const node of nodes) {
    const nodeId = node._id;
    const rec = state.itemNodes.get(nodeId);
    if (!rec) continue;

    payload.items.push({
      id: rec.dbId || 0,
      asset_id: rec.asset_id,
      x_px: node.x(),
      y_px: node.y(),
      rotation_deg: node.rotation(),
      z_index: z++,
      locked: 0,
      meta_json: rec.meta || null
    });
  }

  const r = await fetch(API.saveItems, {
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify(payload)
  });

  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Save failed.');

  // Apply returned id_map (by index)
  if (j.id_map) {
    for (const [idx, newId] of Object.entries(j.id_map)) {
      const i = Number(idx);
      const p = payload.items[i];
      if (!p) continue;

      // Find matching node by position+asset (best-effort)
      // For v1 we’ll just re-load project to sync IDs if needed.
    }
  }

  toast(`Saved. Inserted ${j.inserted}, updated ${j.updated}`);
  log(`Saved items: inserted=${j.inserted}, updated=${j.updated}`);
}

/* ---------------------------
   Selection actions
--------------------------- */
function getSelectedNode() {
  const nodes = state.transformer.nodes();
  return nodes && nodes.length ? nodes[0] : null;
}

function rotateSelected(deg) {
  const node = getSelectedNode();
  if (!node) return;
  node.rotation((node.rotation() + deg) % 360);
  state.itemLayer.draw();
}

function deleteSelected() {
  const node = getSelectedNode();
  if (!node) return;

  // if already saved in DB, delete via save_items deleted_ids (v1 simple: store list and include next save)
  const nodeId = node._id;
  const rec = state.itemNodes.get(nodeId);

  // Immediate remove from canvas
  state.transformer.nodes([]);
  selectionControls(false);
  node.destroy();
  state.itemNodes.delete(nodeId);
  state.itemLayer.draw();
  state.uiLayer.draw();

  // v1: we do not call delete endpoint; we rely on next full save rewriting the set.
  // If you want hard delete now, add deleted_ids tracking + pass to save_items.
  toast('Deleted item (save to persist).');
}

/* ---------------------------
   Export
--------------------------- */
async function exportPNG(type) {
  if (!state.token) return toast('No token.');
  if (!state.bgBitmap) return toast('No background.');

  // BEFORE = background only
  // AFTER  = background + items
  const oldVis = [];
  if (type === 'before') {
    // hide items temporarily
    for (const node of state.itemLayer.getChildren()) {
      oldVis.push([node, node.visible()]);
      node.visible(false);
    }
    state.itemLayer.draw();
  }

  const dataUrl = state.stage.toDataURL({ pixelRatio: 2 });
  const blob = await (await fetch(dataUrl)).blob();

  // restore
  if (type === 'before') {
    for (const [node, vis] of oldVis) node.visible(vis);
    state.itemLayer.draw();
  }

  // upload
  const fd = new FormData();
  fd.append('token', state.token);
  fd.append('export_type', type);
  fd.append('file', blob, `${type}.png`);

  const r = await fetch(API.exportSave, { method:'POST', body: fd });
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Export upload failed.');

  toast(`Export saved: ${type}`);
  log(`${type} export: ${j.file_path}`);
}

/* ---------------------------
   BOM
--------------------------- */
async function fetchBOM() {
  if (!state.token) return toast('No token.');
  const r = await fetch(`${API.bom}?token=${encodeURIComponent(state.token)}`);
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'BOM failed.');
  log('BOM:\n' + JSON.stringify(j.bom, null, 2));
  toast('BOM loaded (see output)');
}

/* ---------------------------
   Background actions: upload, corners, corrected upload, calibration save
--------------------------- */
async function uploadBackground() {
  if (!state.token) return toast('Create/open a project first.');
  const file = ui.bgFile.files && ui.bgFile.files[0];
  if (!file) return toast('Pick a background file.');

  const mode = ui.bgMode.value;

  const fd = new FormData();
  fd.append('token', state.token);
  fd.append('mode', mode);
  fd.append('file', file, file.name);

  const r = await fetch(API.uploadBg, { method:'POST', body: fd });
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Background upload failed.');

  state.backgroundId = Number(j.background_id);
  state.backgroundMode = j.mode;
  state.bgOriginalUrl = j.original_path;
  state.bgCorrectedUrl = '';

  await setBackgroundImage(state.bgOriginalUrl);

  enableBgDependentUI(true);
  ui.btnPickCorners.disabled = (state.backgroundMode !== 'perspective');
  ui.btnCalibrate.disabled = false;
  ui.btnSaveScale.disabled = false;

  // reset corner/cal markers
  clearCornerUI();
  clearCalUI();

  toast('Background uploaded');
  log(`Background uploaded: id=${state.backgroundId}, mode=${state.backgroundMode}`);
}

function startPickCorners() {
  if (!state.bgBitmap) return toast('No background loaded.');
  clearCornerUI();
  state.mode = 'pickCorners';
  ui.btnPickCorners.disabled = true;
  ui.btnUploadCorrected.disabled = true;
  toast('Click corners in order: TL → TR → BR → BL');
}

async function uploadCorrected() {
  if (!state.token) return toast('No token.');
  if (!state.backgroundId) return toast('No background_id.');
  if (state.cornerPoints.length !== 4) return toast('Pick 4 corners first.');

  ui.btnUploadCorrected.disabled = true;

  toast('Warping image…');
  const blob = await warpBackgroundToBlob();

  toast('Uploading corrected…');
  const fd = new FormData();
  fd.append('token', state.token);
  fd.append('background_id', String(state.backgroundId));
  fd.append('corner_points_json', state._lastCornerPointsJson || JSON.stringify(state.cornerPoints));
  fd.append('file', blob, 'corrected.png');

  const r = await fetch(API.saveCorrected, { method:'POST', body: fd });
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Corrected upload failed.');

  state.bgCorrectedUrl = j.corrected_path;

  // Load corrected from server (ensures it matches stored copy)
  await setBackgroundImage(state.bgCorrectedUrl);

  // clear markers
  clearCornerUI();
  toast('Corrected background saved');
  log(`Corrected background: ${state.bgCorrectedUrl}`);
}

function startCalibrate() {
  if (!state.bgBitmap) return toast('No background loaded.');
  clearCalUI();
  state.mode = 'calibrate';
  ui.btnCalibrate.disabled = true;
  ui.btnSaveScale.disabled = true;
  toast('Click two points (known distance)');
}

async function saveScale() {
  if (!state.token) return toast('No token.');
  if (!state.backgroundId) return toast('No background_id.');

  const ppm = parseFloat(ui.pxPerMM.value || '0');
  if (!(ppm > 0)) return toast('Enter/compute px/mm first.');

  state.pxPerMM = ppm;

  const known = parseFloat(ui.knownMM.value || '0');
  const calibration = (state.calPoints.length === 2 && known > 0) ? {
    a: state.calPoints[0],
    b: state.calPoints[1],
    known_mm: known
  } : null;

  const r = await fetch(API.saveCalibration, {
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify({
      token: state.token,
      background_id: state.backgroundId,
      px_per_mm: ppm,
      calibration
    })
  });
  const j = await r.json();
  if (!j.success) throw new Error(j.message || 'Save calibration failed.');

  ui.btnSaveScale.disabled = false;
  toast('Scale saved');
  log(`Scale saved px/mm=${ppm}`);
}

/* ---------------------------
   Wire UI
--------------------------- */
ui.btnCreateProject.addEventListener('click', async () => {
  try { await createProject(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnOpenProject.addEventListener('click', async () => {
  const token = ui.projectToken.value.trim();
  if (!token) return toast('Enter token.');
  try { await openProjectFromToken(token); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnReloadProject.addEventListener('click', async () => {
  try { await reloadProject(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnUploadBg.addEventListener('click', async () => {
  try { await uploadBackground(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.bgMode.addEventListener('change', () => {
  ui.btnPickCorners.disabled = !(state.bgBitmap && ui.bgMode.value === 'perspective');
});

ui.btnPickCorners.addEventListener('click', () => {
  try { startPickCorners(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnUploadCorrected.addEventListener('click', async () => {
  try { await uploadCorrected(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnCalibrate.addEventListener('click', () => startCalibrate());

ui.knownMM.addEventListener('input', () => {
  // If already have 2 points, recompute
  if (state.calPoints.length === 2) {
    const known = parseFloat(ui.knownMM.value || '0');
    if (known > 0) {
      const a = state.calPoints[0], b = state.calPoints[1];
      const px = Math.hypot(b.x-a.x, b.y-a.y);
      const ppm = px / known;
      ui.pxPerMM.value = ppm.toFixed(6);
      state.pxPerMM = ppm;
      ui.btnSaveScale.disabled = false;
    }
  }
});

ui.btnSaveScale.addEventListener('click', async () => {
  try { await saveScale(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnRotateL.addEventListener('click', () => rotateSelected(-15));
ui.btnRotateR.addEventListener('click', () => rotateSelected(15));
ui.btnDeleteSel.addEventListener('click', () => deleteSelected());

ui.btnSaveItems.addEventListener('click', async () => {
  try { await saveItems(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnExportBefore.addEventListener('click', async () => {
  try { await exportPNG('before'); }
  catch (e) { toast(e.message || String(e), 3500); }
});
ui.btnExportAfter.addEventListener('click', async () => {
  try { await exportPNG('after'); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnBOM.addEventListener('click', async () => {
  try { await fetchBOM(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

ui.btnLoadAssets.addEventListener('click', async () => {
  try { await loadAssets(); }
  catch (e) { toast(e.message || String(e), 3500); }
});

// Enable selection transformer when clicking items
state.itemLayer?.on('click tap', (e) => {
  if (e.target && e.target.className === 'Image') {
    state.transformer.nodes([e.target]);
    selectionControls(true);
    state.uiLayer.draw();
  }
});

/* ---------------------------
   Boot: token from URL or localStorage
--------------------------- */
(async function boot() {
  try {
    const url = new URL(location.href);
    const token = url.searchParams.get('token') || localStorage.getItem('panel_designer_token') || '';
    if (token) {
      ui.projectToken.value = token;
      setToken(token);
      await loadAssets().catch(()=>{});
      await reloadProject();
      enableControls(true);
      return;
    }
    await loadAssets().catch(()=>{});
    setStatus('No project');
  } catch (e) {
    toast(e.message || String(e), 3500);
  }
})();